<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists("WOO_SMS_GF_CURL")) {

	class WOO_SMS_GF_CURL
	{

		/**
		 * base url
		 *
		 * @var string
		 */
		private $base_url = '';

		/**
		 * header
		 *
		 * @var array
		 */
		private $headers = array();

		/**
		 * RequestObj
		 *
		 * @var array
		 */
		private $requestObj = array();

		/**
		 * Add your base url and additional headers
		 *
		 * @param string $base_url
		 * @param array $headers
		 * @author Hampus Alstermo <hampus@grafikfabriken.nu>
		 * @version 1.0
		 * @return void
		 */
		public function __construct($base_url, $headers)
		{
			$this->base_url = $base_url;

			if (is_array($headers)) {
				$this->headers = $headers;
			} else {
				$this->headers = array($headers);
			}

			$this->requestObj = array(
				"method" => "GET"
			);
		}


		/**
		 * Sets the basic auth
		 * @version 1.0.0
		 * @date    2017-09-05
		 * @author  Alexander Widén <alexander@grafikfabriken.nu>
		 * @param   string     $username
		 * @param   string     $password
		 * @return  void
		 */
		public function basic_auth($username, $password)
		{
			$this->headers["Authorization"] = 'Basic ' . base64_encode($username . ':' . $password);
		}

		/**
		 * call_api
		 *
		 * @param string $endpoint
		 * @param string $method
		 * @param array $post_data
		 * @param array $query_params
		 * @param boolean $handle_response
		 * @return array response
		 */
		public function callAPI($endpoint, $method = 'GET', $post_data = null, $query_params = null, $handle_response = true)
		{

			$url = $this->base_url . $endpoint;

			// if (is_object($post_data) || is_array($post_data)) $post_data = json_encode($post_data);

			if (!empty($query_params)) $url .= '?' . http_build_query($query_params);

			$this->requestObj = $this->configure_curl(
				$this->requestObj,
				$method,
				$post_data,
				$url
			);

			

			$_response = $this->send($url);

			$response = new stdClass();

			

			if (is_wp_error($_response)) {
				$response->data = array();
				$response->info = array(
					"code" => 500
				);
			} else {
				$response->data = $_response["body"];
				$response->info = $_response["response"];
			}
		

			if (!$handle_response) return $response;

			$data = $this->handle_response($response, $url);
			return $data;
		}



		/**
		 * configureCurl
		 *
		 * @param resource $c cURL session handle
		 * @param string $method
		 * @param array $post_data
		 * @param string $url
		 * @return resource
		 */
		private function configure_curl($ch, $method, $post_data, $url)
		{

			// method – GET
			// timeout – 5 – How long to wait before giving up
			// redirection – 5 – How many times to follow redirections.
			// httpversion – 1.0
			// blocking – true – Should the rest of the page wait to finish loading until this operation is complete?
			// headers – array()
			// body – null
			// cookies – array()

			$this->requestObj["method"] = $method;
			$this->requestObj["timeout"] = 60;
			$this->requestObj["redirection"] = 10;


			if (!empty($this->headers)) {
				$this->requestObj["headers"] = $this->headers;
			}

			switch ($method) {
				case 'POST':
				case 'PUT':
					$this->requestObj["method"] = $method;
					$this->requestObj["body"] = $post_data;
					break;
				case 'GET':
					$this->requestObj["method"] = "GET";
					break;
				case 'DELETE':
					$this->requestObj["method"] = "DELETE";
					break;

				default:
					throw new Exception('Method ' . $method . ' not recognized.');
			}

			if ($_SERVER['HTTP_HOST'] == 'localhost' || strpos($_SERVER['HTTP_HOST'], '.test') !== false) {
				$this->requestObj["sslverify"] = false;
			}

			

			return apply_filters('WOO_SMS_GF_CURL_modify_options', $this->requestObj);
		}

		/**
		 * send
		 *
		 * @param resource $c cURL session handle
		 * @return WP_Error|Array
		 */
		private function send($url)
		{
			return wp_remote_request($url, $this->requestObj);
		}

		/**
		 * Converts an SimpleXMLElement to an array
		 * @version 1.0.0
		 * @date    2017-08-25
		 * @author  Alexander Widén <alexander@grafikfabriken.nu>
		 * @param   SimpleXMLElement $parent
		 * @return 	array
		 */
		private function XML2Array(SimpleXMLElement $parent)
		{
			$array = array();

			foreach ($parent as $name => $element) {
				($node = &$array[$name])
					&& (1 === count($node) ? $node = array($node) : 1)
					&& $node = &$node[];

				$node = $element->count() ? $this->XML2Array($element) : trim($element);
			}

			return $array;
		}

		/**
		 * handleResponse
		 *
		 * @param Object $response
		 * @param array $response_info
		 * @param string $url
		 * @return mixed
		 */
		private function handle_response($response, $url)
		{

			if (is_wp_error($response)) {
				throw new Exception('Cannot connect to the API');
			}

			$response_body = $response["body"];
			$response_info = $response["response"];

			$is_xml = @simplexml_load_string($response_body);
			if ($is_xml) {
				$xml   = simplexml_load_string($response_body);
				$array = $this->XML2Array($xml);
				$response = array($xml->getName() => $array);
			} else {
				$response = json_decode($response_body);
			}

			switch ($response_info['code']) {
				case 0:
					throw new Exception('TIMEOUT');
					break;
				case 200:
					return $response;
					break;
				case 302:
					return $response;
					break;
				case 400:
					throw new Exception('Bad Request');
					break;
				case 500:
					throw new Exception('Internal Server Error');
					break;
				default:
					throw new Exception('Unable to connect to the api (' . $url . '). Response code: ' . $response_info['code']);
			}

			return $response;
		}


		/**
		 * Deserialize a JSON string into an object.
		 * @param object $data object or primitive to be deserialized
		 * @param string $class class name is passed as a string
		 * @return object an instance of $class
		 */
		public function deserialize($data, $class)
		{
			if ($data == null) {
				$deserialized = null;
			} elseif (substr($class, 0, 6) == 'array[') {
				$subClass = substr($class, 6, -1);
				$values = array();
				if (wooSMSarray_has_items($data)) {
					foreach ($data as $value) {
						$values[] = $this->deserialize($value, $subClass);
					}
				}
				$deserialized = $values;
			} elseif ('\DateTime' == $class) {
				$deserialized = new \DateTime('@' . ($data / 1000), new \DateTimeZone('UTC'));
			} elseif ('\DT' == $class) {
				$deserialized = new \DateTime($data, new \DateTimeZone('UTC'));
			} elseif ('time' == $class) { // @todo ADDED
				$deserialized = date_i18n("Y-m-d H:i:s", strtotime($data));
			} elseif (in_array($class, array('string', 'int', 'float', 'bool', 'time'))) {
				settype($data, $class);
				$deserialized = $data;
			} else {
				$instance = new $class();
				foreach ($instance->dataTypes as $property => $type) {
					if (isset($data->$property)) {
						$instance->$property = $this->deserialize($data->$property, $type);
					}
				}
				$deserialized = $instance;
			}

			return $deserialized;
		}
	}
}
