<?php
if(!defined('ABSPATH')){
    exit; // Exit if accsessd directly
}

if (!class_exists('Woo_SMS_REST_Controller')) {


    /**
     * Woo SMS Rest Controller
     *
     * @author Hampus Alstermo <hampus@grafikfabriken.nu>
     * @version 1.0
     */
    class Woo_SMS_REST_Controller extends WOO_SMS_Singleton
    {

        /**
         * Is sandbox?
         *
         * @var boolean
         */
        public $sandbox = false;

        /**
         * The rest url
         *
         * @var string
         */
        public $rest_url = 'https://api.smsfabriken.se/';

        /**
         * The rest base
         *
         * @var string
         */
        public $rest_base = 'wp-json/gf-woosms-handler/v1';

        /**
         * Buy base
         *
         * @var string
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public $buy_base = 'checkout/buy/';

        /**
         * Rest end points
         *
         * @var array
         */
        public $rest_endpoints = array(
            "add_license" => '/license/add/',
            "check_license" => '/license/check/',
            "send_sms" => '/send/sms/'
        );
        
        /**
         * Construct function
         *
         * @return void
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function _construct(){

            if($this->sandbox){
                $this->rest_url = 'https://api.woosms.test/';
            }

            add_action('wp_ajax_woo_sms_gf_send_sms', array($this,'ajax_send_sms'));
            add_action('wp_ajax_woo_sms_gf_send_bulk_sms', array($this,'ajax_send_bulk_sms'));

        }

        /**
         * Send single sms trough AJAX
         *
         * @return void
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function ajax_send_bulk_sms(){

            if(is_user_logged_in() && current_user_can('administrator')){

                $to = isset($_POST["to"]) ? $this->sanitize_array($_POST["to"]) : null;
                $from = isset($_POST["from"]) ? sanitize_text_field($_POST["from"]) : null;
                $message = isset($_POST["message"]) ? sanitize_text_field($_POST["message"]) : null;

                if ($to !== null && $from !== null && $message !== null) {

                    $option = wooSMSController()->get_options();
                    $sms_sent = wooSMSRestController()->send_sms($from, $to, $message, $option->get_license_key());

                    if (is_a($sms_sent, 'stdClass') && isset($sms_sent->used_credits)) {
                        wp_send_json_success(array("used_credits" => $sms_sent->used_credits, "credits" => $sms_sent->license->credits));
                    }
                    
                }

                
            }

            wp_redirect('/',500);
            wp_send_json_error();
            die();

        }

        /**
         * Sanitize simple array
         *
         * @param array $value
         * @return array
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         * @date [date]
         * @since [since]
         */
        public function sanitize_array($value){

            if(wooSMSarray_has_items($value)){
                
                $res = [];
                foreach($value as $v){
                    $res[] = filter_var($v,FILTER_SANITIZE_STRING);
                }
                
                return $res;
            }

            return array();
        }

        /**
         * Send single sms trough AJAX
         *
         * @return void
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function ajax_send_sms(){

            if(is_user_logged_in() && current_user_can('administrator')){

                $to = isset($_POST["to"]) ? sanitize_text_field($_POST["to"]) : null;
                $from = isset($_POST["from"]) ? sanitize_text_field($_POST["from"]) : null;
                $message = isset($_POST["message"]) ? sanitize_text_field($_POST["message"]) : null;

                if ($to !== null && $from !== null && $message !== null) {

                    $option = wooSMSController()->get_options();
                    $sms_sent = wooSMSRestController()->send_sms($from, $to, $message, $option->get_license_key());
                    

                    if (is_a($sms_sent, 'stdClass') && isset($sms_sent->used_credits)) {
                        wp_send_json_success(array("used_credits" => $sms_sent->used_credits, "credits" => $sms_sent->license->credits));
                    }
                    
                }

                
            }

            wp_redirect('/',500);
            wp_send_json_error();
            die();

        }

        /**
         * Genereate a new license
         *
         * @return void
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function generate_license(){

            $ep = $this->get_endpoint('add_license');
            
            $curl = new WOO_SMS_GF_CURL($this->get_rest_base_url(),$this->get_headers());

            $response = $this->handle_response($curl->callAPI($this->get_endpoint('add_license') . "?domain=" . get_home_url(),'GET',null,null,false));

            if ($response !== false && is_a($response,'stdClass')) {
                $option = wooSMSController()->get_options();
                $option->license_key = $response->key;
                $option->credits = $response->credits;
                $option->save();

                wooSMSController()->info["welcome"] = sprintf(__('Hello and welcome to SMSFabriken. You have %s text messages to try it out. Your license key is: %s. All purchases you make are tied to your license key.', 'woocommerce-sms-gf'), $option->credits, $option->get_license_key());

            }else{
                
                wooSMSController()->errors["no_key"] = __('SMSFabriken could not generate a license key for you','woocommerce-sms-gf');
            }

        }

        /**
         * Send SMS
         *
         * @param string $from
         * @param string|array $recipients
         * @param string $template
         * @param string $license_key
         * @return stdClass|boolean
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function send_sms($from, $recipients, $template, $license_key){

            //Instansiate new Curl
            $curl = new WOO_SMS_GF_CURL($this->get_rest_base_url(), $this->get_headers());
            
            //Hook endpoint
            $ep = $this->get_endpoint('send_sms') . $license_key;

            //Setup post data
            $post_data = array(
                "from" => $from,
                "recipients" => is_array($recipients) ? $recipients : array($recipients),
                "message" => (string) $template,   
            );

            //Doo cURL
            return $this->handle_response($curl->callAPI($ep,'POST',$post_data,null,false));

        }

        /**
         * Check license!
         * 
         * @param string $license_key
         * @return void
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function check_license($license_key){

            $ep = $this->get_endpoint('check_license');
            
            $curl = new WOO_SMS_GF_CURL($this->get_rest_base_url(),$this->get_headers());
            $response = $this->handle_response($curl->callAPI($ep . $license_key ,'GET',null,null,false));
            
            if ($response !== false) {

                $option = wooSMSController()->get_options();
                $option->license_key = $response->key;
                $option->credits = (int) $response->credits;
                $option->save();

            }else{
                wooSMSController()->errors["invalid_key"] = __(sprintf('SMSFabriken could not verify your license key: <strong>%s</strong>. Contact support.',$license_key),'woocommerce-sms-gf');

            }
        }

        /**
		 * Converts an SimpleXMLElement to an array
		 * @version 1.0.0
		 * @date    2017-08-25
		 * @author  Alexander Widén <alexander@grafikfabriken.nu>
		 * @param   SimpleXMLElement $parent
		 * @return 	array
		 */
		private function XML2Array(SimpleXMLElement $parent){
		    $array = array();

		    foreach ($parent as $name => $element) {
		        ($node = & $array[$name])
		            && (1 === count($node) ? $node = array($node) : 1)
		            && $node = & $node[];

		        $node = $element->count() ? $this->XML2Array($element) : trim($element);
		    }

		    return $array;
		}

        /**
		 * Handle curl response
		 *
		 * @param Object $response
		 * @param array $response_info
		 * @param string $url
		 * @return mixed
		 */
		private function handle_response( $curl){

            $response = $curl->data;
            $response_info = $curl->info;
            
            $response = json_decode( $response );
            
			switch($response_info['code']) {
				case 200: 	return $response; break;
                default: return false;
			}

			return $response;
        }

        /**
         * Get buy url
         *
         * @return string
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function get_buy_url(){
            return $this->rest_url . $this->buy_base;
        }

        /**
         * Get headers
         *
         * @return array
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function get_headers(){
            return array('Content-Type: application/json');
        }

        /**
         * Get rest base url
         *
         * @return string
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function get_rest_base_url(){
            return $this->rest_url . $this->rest_base;
        }

        /**
         * Get full endpoint
         *
         * @param string $ep
         * @return string
         * @author Hampus Alstermo <hampus@grafikfabriken.nu>
         * @version 1.0
         */
        public function get_endpoint($ep){
            return isset($this->rest_endpoints[$ep]) ? $this->rest_endpoints[$ep] : false;
        }

    }

    /**
     * wrapper function for Woo_SMS_REST_Controller
     *
     * @return Woo_SMS_REST_Controller
     * @author Hampus Alstermo <hampus@grafikfabriken.nu>
     * @version 1.0
     */
    function wooSMSRestController(){
        return Woo_SMS_REST_Controller::getInstance();
    }

    wooSMSRestController(); // Run controller

}